# ========= Copyright 2023-2024 @ CAMEL-AI.org. All Rights Reserved. =========
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
# ========= Copyright 2023-2024 @ CAMEL-AI.org. All Rights Reserved. =========
from unittest.mock import MagicMock, patch

import pytest

from camel.messages import BaseMessage
from camel.models import BaseModelBackend, OpenAIAudioModels
from camel.toolkits.video_analysis_toolkit import VideoAnalysisToolkit


# Mock ffmpeg.Error for testing
class MockFFmpegError(Exception):
    def __init__(self, msg="FFmpeg error"):
        self.stderr = msg.encode() if isinstance(msg, str) else msg
        super().__init__(msg)


# Fixtures
@pytest.fixture
def mock_model():
    return MagicMock(spec=BaseModelBackend)


@pytest.fixture
def mock_audio_models():
    mock_audio = MagicMock(spec=OpenAIAudioModels)
    mock_audio.speech_to_text.return_value = "This is a test transcription"
    return mock_audio


@pytest.fixture
def mock_video_toolkit(mock_model, mock_audio_models):
    r"""Create a VideoAnalysisToolkit with mocked dependencies"""
    with (
        patch("tempfile.mkdtemp", return_value="/tmp/mock_dir"),
        patch("pathlib.Path.mkdir"),
        patch("camel.agents.ChatAgent"),
        patch("camel.toolkits.video_analysis_toolkit.VideoDownloaderToolkit"),
        patch(
            "camel.models.OpenAIAudioModels", return_value=mock_audio_models
        ),
        patch("os.path.exists", return_value=True),
        patch("shutil.rmtree"),
        patch("ffmpeg.input"),
        patch("ffmpeg.probe"),
    ):
        # Mock the chat agent
        mock_agent = MagicMock()
        mock_agent.step.return_value.msgs = [
            MagicMock(content="Test response")
        ]

        # Patch the ChatAgent import
        with patch("camel.agents.ChatAgent", return_value=mock_agent):
            toolkit = VideoAnalysisToolkit(
                model=mock_model, use_audio_transcription=True
            )
            # Manually set audio models to ensure it's available
            toolkit.audio_models = mock_audio_models
            toolkit._use_audio_transcription = True
            toolkit.vl_agent = mock_agent
            yield toolkit


def test_init_default():
    r"""Test initialization with default parameters"""
    with (
        patch("tempfile.mkdtemp", return_value="/tmp/mock_dir"),
        patch("pathlib.Path.mkdir"),
        patch("camel.agents.ChatAgent"),
        patch("camel.toolkits.video_analysis_toolkit.VideoDownloaderToolkit"),
    ):
        toolkit = VideoAnalysisToolkit()
        assert toolkit._cleanup is True
        assert toolkit._use_audio_transcription is False
        # Use startswith instead of exact match to handle /private/tmp on macOS
        assert toolkit._download_directory.as_posix().endswith("/mock_dir")


def test_init_with_download_directory():
    r"""Test initialization with custom download directory"""
    with (
        patch("pathlib.Path.mkdir"),
        patch("camel.agents.ChatAgent"),
        patch("camel.toolkits.video_analysis_toolkit.VideoDownloaderToolkit"),
        patch("os.path.exists", return_value=True),
    ):
        toolkit = VideoAnalysisToolkit(download_directory="/custom/dir")
        assert toolkit._cleanup is False
        assert toolkit._download_directory.as_posix() == "/custom/dir"


def test_init_with_audio_transcription():
    r"""Test initialization with audio transcription enabled"""
    mock_audio = MagicMock()
    with (
        patch("tempfile.mkdtemp", return_value="/tmp/mock_dir"),
        patch("pathlib.Path.mkdir"),
        patch("camel.agents.ChatAgent"),
        patch("camel.toolkits.video_analysis_toolkit.VideoDownloaderToolkit"),
        # Mock OpenAIAudioModels and prevent the exception
        patch("camel.models.OpenAIAudioModels", return_value=mock_audio),
        # Prevent the exception from being raised during initialization
        patch(
            "camel.toolkits.video_analysis_toolkit.OpenAIAudioModels.__init__",
            return_value=None,
        ),
    ):
        toolkit = VideoAnalysisToolkit(use_audio_transcription=True)
        # Manually set the audio models and use_audio_transcription flag
        toolkit.audio_models = mock_audio
        toolkit._use_audio_transcription = True
        assert toolkit._use_audio_transcription is True
        assert toolkit.audio_models is mock_audio


def test_init_with_invalid_directory():
    r"""Test initialization with invalid directory"""
    with (
        patch(
            "pathlib.Path.mkdir", side_effect=ValueError("Invalid directory")
        ),
        patch("camel.agents.ChatAgent"),
    ):
        with pytest.raises(ValueError):
            VideoAnalysisToolkit(download_directory="invalid_dir")


# Test cleanup
def test_cleanup():
    r"""Test cleanup on object destruction"""
    with (
        patch("tempfile.mkdtemp", return_value="/tmp/mock_dir"),
        patch("pathlib.Path.mkdir"),
        patch("camel.agents.ChatAgent"),
        patch("camel.toolkits.video_analysis_toolkit.VideoDownloaderToolkit"),
        patch("os.path.exists", return_value=True),
        patch("os.remove") as mock_remove,
        patch("shutil.rmtree") as mock_rmtree,
    ):
        toolkit = VideoAnalysisToolkit()
        # Add a temp file to clean up
        toolkit._temp_files.append("/tmp/test.mp3")
        # Trigger __del__
        toolkit.__del__()

        mock_remove.assert_called_once_with("/tmp/test.mp3")
        # Use any_call instead of assert_called_once_with to handle PosixPath
        assert any(
            call.args[0].as_posix().endswith("/mock_dir")
            for call in mock_rmtree.mock_calls
        )


# Test audio extraction
def test_extract_audio_from_video(mock_video_toolkit):
    r"""Test extracting audio from video"""
    mock_ffmpeg = MagicMock()
    with patch("ffmpeg.input", return_value=mock_ffmpeg) as mock_input:
        mock_ffmpeg.output.return_value = mock_ffmpeg

        result = mock_video_toolkit._extract_audio_from_video(
            "/path/to/video.mp4"
        )

        assert result == "/path/to/video.mp3"
        assert "/path/to/video.mp3" in mock_video_toolkit._temp_files
        mock_input.assert_called_once_with("/path/to/video.mp4")
        mock_ffmpeg.output.assert_called_once()
        mock_ffmpeg.run.assert_called_once_with(quiet=True)


def test_extract_audio_error(mock_video_toolkit):
    r"""Test handling errors in audio extraction"""
    mock_ffmpeg = MagicMock()
    with patch("ffmpeg.input", return_value=mock_ffmpeg):
        mock_ffmpeg.output.return_value = mock_ffmpeg

        # Create a patch for ffmpeg.Error at the import level
        with patch("ffmpeg.Error", MockFFmpegError):
            # Set up the side effect after patching ffmpeg.Error
            mock_ffmpeg.run.side_effect = MockFFmpegError("FFmpeg failed")

            with pytest.raises(RuntimeError):
                mock_video_toolkit._extract_audio_from_video(
                    "/path/to/video.mp4"
                )


# Test audio transcription
def test_transcribe_audio(mock_video_toolkit, mock_audio_models):
    r"""Test audio transcription"""
    result = mock_video_toolkit._transcribe_audio("/path/to/audio.mp3")

    assert result == "This is a test transcription"
    mock_audio_models.speech_to_text.assert_called_once_with(
        "/path/to/audio.mp3"
    )


def test_transcribe_audio_disabled(mock_video_toolkit):
    r"""Test when audio transcription is disabled"""
    mock_video_toolkit._use_audio_transcription = False

    result = mock_video_toolkit._transcribe_audio("/path/to/audio.mp3")

    assert result == "No audio transcription available."


def test_transcribe_audio_error(mock_video_toolkit, mock_audio_models):
    r"""Test handling errors in audio transcription"""
    mock_audio_models.speech_to_text.side_effect = Exception(
        "Transcription error"
    )

    result = mock_video_toolkit._transcribe_audio("/path/to/audio.mp3")

    assert result == "Audio transcription failed."


# Test keyframe extraction
def test_extract_keyframes_with_scenes(mock_video_toolkit):
    r"""Test extracting keyframes with scene detection"""
    mock_scene_manager = MagicMock()
    mock_video_manager = MagicMock()
    mock_scene = MagicMock()
    mock_scene_list = [(10.0, 20.0), (30.0, 40.0)]
    mock_frame = MagicMock()

    with (
        patch("scenedetect.SceneManager", return_value=mock_scene_manager),
        patch("scenedetect.VideoManager", return_value=mock_video_manager),
        patch(
            "scenedetect.detectors.ContentDetector", return_value=mock_scene
        ),
        # Directly patch the function in the module where it's being called
        # from
        patch(
            "camel.toolkits.video_analysis_toolkit._capture_screenshot",
            return_value=mock_frame,
        ),
    ):
        mock_scene_manager.get_scene_list.return_value = mock_scene_list

        result = mock_video_toolkit._extract_keyframes("/path/to/video.mp4", 2)

        assert len(result) == 2
        assert result[0] == mock_frame
        assert result[1] == mock_frame


def test_extract_keyframes_no_scenes(mock_video_toolkit):
    r"""Test extracting keyframes when no scenes are detected"""
    mock_scene_manager = MagicMock()
    mock_video_manager = MagicMock()
    mock_scene = MagicMock()
    mock_cap = MagicMock()
    mock_frame = MagicMock()

    with (
        patch("scenedetect.SceneManager", return_value=mock_scene_manager),
        patch("scenedetect.VideoManager", return_value=mock_video_manager),
        patch(
            "scenedetect.detectors.ContentDetector", return_value=mock_scene
        ),
        patch("cv2.VideoCapture", return_value=mock_cap),
        # Directly patch the function in the module where it's being called
        # from
        patch(
            "camel.toolkits.video_analysis_toolkit._capture_screenshot",
            return_value=mock_frame,
        ),
    ):
        mock_scene_manager.get_scene_list.return_value = []
        mock_cap.get.side_effect = [100, 25]  # total_frames, fps

        result = mock_video_toolkit._extract_keyframes("/path/to/video.mp4", 2)

        assert len(result) == 2
        assert result[0] == mock_frame
        assert result[1] == mock_frame


def test_extract_keyframes_invalid_num_frames(mock_video_toolkit):
    r"""Test extracting keyframes with invalid number of frames"""
    mock_scene_manager = MagicMock()
    mock_video_manager = MagicMock()
    mock_scene = MagicMock()
    mock_scene_list = [(10.0, 20.0)]
    mock_frame = MagicMock()

    with (
        patch("scenedetect.SceneManager", return_value=mock_scene_manager),
        patch("scenedetect.VideoManager", return_value=mock_video_manager),
        patch(
            "scenedetect.detectors.ContentDetector", return_value=mock_scene
        ),
        # Directly patch the function in the module where it's being called
        # from
        patch(
            "camel.toolkits.video_analysis_toolkit._capture_screenshot",
            return_value=mock_frame,
        ),
    ):
        mock_scene_manager.get_scene_list.return_value = mock_scene_list

        result = mock_video_toolkit._extract_keyframes(
            "/path/to/video.mp4", -1
        )

        assert len(result) == 1
        assert result[0] == mock_frame


# Test ask_question_about_video
def test_ask_question_about_video_local_file(mock_video_toolkit):
    r"""Test asking a question about a local video file"""
    mock_frames = [MagicMock(), MagicMock()]

    with (
        patch.object(
            mock_video_toolkit, "_extract_keyframes", return_value=mock_frames
        ),
        patch("os.path.exists", return_value=True),
    ):
        # Directly patch the _transcribe_audio method
        mock_video_toolkit._transcribe_audio = MagicMock(
            return_value="Test transcription"
        )

        result = mock_video_toolkit.ask_question_about_video(
            "/path/to/video.mp4", "What is happening in this video?"
        )

        assert result == "Test response"
        mock_video_toolkit.vl_agent.step.assert_called_once()

        # Check that the message was created with the right content
        call_args = mock_video_toolkit.vl_agent.step.call_args[0][0]
        assert isinstance(call_args, BaseMessage)
        assert "Test transcription" in call_args.content
        assert "What is happening in this video?" in call_args.content
        assert call_args.image_list == mock_frames


def test_ask_question_about_video_url(mock_video_toolkit):
    r"""Test asking a question about a video URL"""
    mock_frames = [MagicMock(), MagicMock()]
    mock_video_toolkit.video_downloader_toolkit.download_video.return_value = (
        "/path/to/downloaded.mp4"
    )

    with (
        patch.object(
            mock_video_toolkit, "_extract_keyframes", return_value=mock_frames
        ),
        patch("os.path.exists", return_value=True),
    ):
        # Directly patch the _transcribe_audio method
        mock_video_toolkit._transcribe_audio = MagicMock(
            return_value="Test transcription"
        )

        result = mock_video_toolkit.ask_question_about_video(
            "https://example.com/video.mp4", "What is happening in this video?"
        )

        assert result == "Test response"
        mock_video_toolkit.video_downloader_toolkit.download_video.assert_called_once_with(
            "https://example.com/video.mp4"
        )


def test_ask_question_about_video_no_audio(mock_video_toolkit):
    r"""Test asking a question about a video with audio transcription
    disabled.
    """
    mock_frames = [MagicMock(), MagicMock()]
    mock_video_toolkit._use_audio_transcription = False

    with (
        patch.object(
            mock_video_toolkit, "_extract_keyframes", return_value=mock_frames
        ),
        patch("os.path.exists", return_value=True),
    ):
        result = mock_video_toolkit.ask_question_about_video(
            "/path/to/video.mp4", "What is happening in this video?"
        )

        assert result == "Test response"
        # Check that the message was created with the right content
        call_args = mock_video_toolkit.vl_agent.step.call_args[0][0]
        assert "No audio transcription available" in call_args.content


def test_ask_question_about_video_empty_question(mock_video_toolkit):
    r"""Test asking an empty question"""
    with pytest.raises(ValueError):
        mock_video_toolkit.ask_question_about_video("/path/to/video.mp4", "")


def test_ask_question_about_video_file_not_found(mock_video_toolkit):
    r"""Test handling file not found error"""
    with patch("os.path.exists", return_value=False):
        result = mock_video_toolkit.ask_question_about_video(
            "/path/to/nonexistent.mp4", "What is happening in this video?"
        )

        assert "Error" in result
        assert "not found" in result


def test_ask_question_about_video_download_error(mock_video_toolkit):
    r"""Test handling video download error"""
    mock_video_toolkit.video_downloader_toolkit.download_video.return_value = (
        None
    )

    result = mock_video_toolkit.ask_question_about_video(
        "https://example.com/video.mp4", "What is happening in this video?"
    )

    assert "Error" in result
    assert "Failed to download" in result


def test_ask_question_about_video_keyframe_error(mock_video_toolkit):
    r"""Test handling keyframe extraction error"""
    with (
        patch.object(
            mock_video_toolkit,
            "_extract_keyframes",
            side_effect=ValueError("Keyframe error"),
        ),
        patch("os.path.exists", return_value=True),
    ):
        result = mock_video_toolkit.ask_question_about_video(
            "/path/to/video.mp4", "What is happening in this video?"
        )

        assert "Error" in result
        assert "Keyframe error" in result


def test_ask_question_about_video_empty_response(mock_video_toolkit):
    r"""Test handling empty response from model"""
    mock_video_toolkit.vl_agent.step.return_value.msgs = []
    mock_frames = [MagicMock(), MagicMock()]

    with (
        patch.object(
            mock_video_toolkit, "_extract_keyframes", return_value=mock_frames
        ),
        patch("os.path.exists", return_value=True),
    ):
        # Directly patch the _transcribe_audio method
        mock_video_toolkit._transcribe_audio = MagicMock(
            return_value="Test transcription"
        )

        result = mock_video_toolkit.ask_question_about_video(
            "/path/to/video.mp4", "What is happening in this video?"
        )

        assert "Failed to generate an answer" in result


# Test get_tools
def test_get_tools(mock_video_toolkit):
    r"""Test getting toolkit tools"""
    tools = mock_video_toolkit.get_tools()

    assert len(tools) == 1
    assert tools[0].func == mock_video_toolkit.ask_question_about_video
